﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Diagnostics;
using Web_Files_Enhanced.Properties;
using System.ComponentModel;

namespace Web_Files_Enhanced
{
    public partial class MainForm : Form
    {
        private List<string> _webFilePaths = new List<string>();
        private int _editPathIndex = -1, _previewPathIndex = -1;
        private string _siteDirectory;

        public MainForm()
        {
            InitializeComponent();
            LoadSettingsAndDefaults();
        }

        private void LoadSettingsAndDefaults()
        {
            splitContainer.Orientation = Settings.Default.SplitViewOrient;
            txtEditor.Font = Settings.Default.EditorFont;

            if (Settings.Default.IsFirstBoot)
            {
                this.CenterToScreen();
                Settings.Default.IsFirstBoot = false;
            }
            else if (this.WindowState != FormWindowState.Maximized)
            {
                this.Bounds = Settings.Default.FormBounds;
            }

            if (Settings.Default.SplitterDistance == 0)
            {
                Settings.Default.SplitterDistance = splitContainer.ClientSize.Width / 2;
            }

            splitContainer.SplitterDistance = Settings.Default.SplitterDistance;
        }

        protected override void OnClosing(CancelEventArgs e)
        {
            SaveFile();
            Settings.Default.SplitViewOrient = splitContainer.Orientation;
            Settings.Default.FormState = this.WindowState;
            Settings.Default.SplitterDistance = splitContainer.SplitterDistance;

            if (this.WindowState == FormWindowState.Normal)
            {
                Settings.Default.FormBounds = this.Bounds;
            }
            else
            {
                Settings.Default.FormBounds = this.RestoreBounds;
            }

            Settings.Default.Save();
            base.OnClosing(e);
        }

        protected override void OnKeyDown(KeyEventArgs e)
        {
            base.OnKeyDown(e);

            if (e.KeyCode == Keys.F5)
            {
                SaveFile();
                UpdatePreview();
            }
        }

        private void LoadWebSite(string sitePath)
        {
            string[] filePaths = null;

            try
            {
                filePaths = Directory.GetFiles(sitePath);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, Application.ProductName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            _siteDirectory = sitePath;

            for (int i = 0; i < filePaths.Length; i++)
            {
                string ext = Path.GetExtension(filePaths[i]).ToLower();
                //if it is a plain text web file, then add safeFileName to
                // edit file comboBox and _webFiles List
                if (ext == ".css" || ext == ".html" || ext == ".htm")
                {
                    comboEditFile.Items.Add(Path.GetFileName(filePaths[i]));
                    comboPreviewFile.Items.Add(Path.GetFileName(filePaths[i]));
                    _webFilePaths.Add(filePaths[i]);
                }
            }

            SetDefaultPage(filePaths);
        }

        private void SetDefaultPage(string[] fileNames)
        {
            int index = 0;

            for (int i = 0; i < fileNames.Length; i++)
            {
                string fileName = Path.GetFileNameWithoutExtension(fileNames[i]).ToLower();
                //If there is an index file, set it as the default viewed page
                if (fileName == "index")
                {
                    index = i;
                    break;
                }
            }

            _editPathIndex = _previewPathIndex = index;
            comboEditFile.SelectedIndex = index;
            comboPreviewFile.SelectedIndex = index;
            UpdateEditor();
            UpdatePreview();
            SetControlAbility(true);
        }

        /// <summary>
        /// Enables or disables controls that operate on website files
        /// </summary>
        private void SetControlAbility(bool isEnabled)
        {
            btnCloseSite.Enabled = isEnabled;
            btnNewFile.Enabled = isEnabled;
            txtEditor.Enabled = isEnabled;
            comboEditFile.Enabled = isEnabled;
            comboPreviewFile.Enabled = isEnabled;
            tsmiZoomFactor.Enabled = isEnabled;
            btnExploreSite.Enabled = isEnabled;
        }

        /// <summary>
        /// Reads the web document and updates the textBoxes text accordingly
        /// </summary>
        private void UpdateEditor()
        {
            txtEditor.Text = File.ReadAllText(_webFilePaths[_editPathIndex]);
        }

        /// <summary>
        /// Writes any chages to the selected edit file
        /// </summary>
        private void SaveFile()
        {
            if (_editPathIndex != -1)
            {
                string fileName = _webFilePaths[_editPathIndex];

                if (File.Exists(fileName))
                {
                    File.WriteAllText(fileName, txtEditor.Text);
                }
            }
        }

        /// <summary>
        /// Navigates the webBrowser control to the selected preview file
        /// </summary>
        private void UpdatePreview()
        {
            webBrowser.Navigate(_webFilePaths[_previewPathIndex]);
        }

        private void UncheckAllZoomItems()
        {
            foreach (ToolStripMenuItem item in tsmiZoomFactor.DropDownItems)
            {
                item.Checked = false;
            }
        }

        /// <summary>
        /// Reset any values related to the website
        /// </summary>
        private void CloseWebSite()
        {
            comboEditFile.Items.Clear();
            comboPreviewFile.Items.Clear();
            _webFilePaths.Clear();
            txtEditor.Clear();
            //Navigate away from absent page
            webBrowser.Navigate(string.Empty);
            //Reset _siteDirectory to null
            _siteDirectory = null;
            //Disable webSite file related controls
            SetControlAbility(false);
        }

        #region Event handlers
        private void btnOpenDir_Click(object sender, EventArgs e)
        {
            using (FolderBrowserDialog folderBrowserDialog = new FolderBrowserDialog())
            {
                if (Settings.Default.LastFolder.Length > 0)
                {
                    folderBrowserDialog.SelectedPath = Settings.Default.LastFolder;
                }

                if (folderBrowserDialog.ShowDialog() == DialogResult.OK)
                {
                    try
                    {
                        LoadWebSite(folderBrowserDialog.SelectedPath);
                        Settings.Default.LastFolder = folderBrowserDialog.SelectedPath;
                    }
                    catch (Exception ex)
                    {
                        CloseWebSite();
                        MessageBox.Show(ex.Message, Application.ProductName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }

            }
        }

        private void comboSelectEditFile_SelectedIndexChanged(object sender, EventArgs e)
        {
            _editPathIndex = comboEditFile.SelectedIndex;
            UpdateEditor();
        }

        private void comboSelectPreviewFile_SelectedIndexChanged(object sender, EventArgs e)
        {
            _previewPathIndex = comboPreviewFile.SelectedIndex;
            UpdatePreview();
        }

        private void btnCloseWebSite_Click(object sender, EventArgs e)
        {
            CloseWebSite();
        }

        private void btnNewFile_Click(object sender, EventArgs e)
        {
            using (NewFileForm dlgNewFile = new NewFileForm(_siteDirectory))
            {
                //Open the form at the cursor position for convienience
                dlgNewFile.Location = Cursor.Position;

                if (dlgNewFile.ShowDialog() == DialogResult.OK)
                {
                    //Reopen WebSite, easiest way to add file to project
                    string temp = _siteDirectory;
                    CloseWebSite();
                    LoadWebSite(temp);
                }
            }
        }

        private void btnSettings_Click(object sender, EventArgs e)
        {
            using (SettingsForm frmSettings = new SettingsForm(this))
            {
                if (frmSettings.ShowDialog() == DialogResult.OK)
                {
                    txtEditor.Font = Settings.Default.EditorFont;
                }
            }
        }

        private void ZoomItems_Click(object sender, EventArgs e)
        {
            ToolStripMenuItem current = ((ToolStripMenuItem)sender);
            // We are using the tag property to store the int
            int zoomLevel = int.Parse(current.Tag.ToString());
            webBrowser.Zoom(zoomLevel);
            UncheckAllZoomItems();
            current.Checked = true;
        }

        private void btnExplore_Click(object sender, EventArgs e)
        {
            Process.Start(_siteDirectory);
        }

        private void ToolBarButton_MouseLeave(object sender, EventArgs e)
        {
            lblCaption.Text = string.Empty;
        }

        private void ToolBarButton_MouseEnter(object sender, EventArgs e)
        {
            RolloverToolStripButton button = (RolloverToolStripButton)sender;
            lblCaption.Text = button.Text;
        }
        #endregion

        public TextBox TextEditor
        {
            get { return txtEditor; }
        }

        public SplitContainer SplitView
        {
            get { return splitContainer; }
        }
    }
}
